package
{
	import alternativa.engine3d.controllers.CameraController;
	import alternativa.engine3d.core.Camera3D;
	import alternativa.engine3d.core.Object3D;
	import alternativa.engine3d.core.Scene3D;
	import alternativa.engine3d.display.View;
	import alternativa.types.Point3D;
	import alternativa.utils.FPS;

	import flash.display.StageAlign;
	import flash.display.StageScaleMode;
	import flash.events.Event;

	import mx.collections.ArrayCollection;
	import mx.core.Application;
	import mx.core.UIComponent;

	/**
	 * 	The EngineManager holds all of the code related to maintaining the Alternativa 3D engine.
	 */
	public class EngineManager extends UIComponent
	{
		public var scene:Scene3D;
		public var view:View;
		public var camera:Camera3D;
		public var cameraController:CameraController;

		// a collection of the BaseObjects
		protected var baseObjects:ArrayCollection = new ArrayCollection();
		// a collection where new BaseObjects are placed, to avoid adding items
		// to baseObjects while in the baseObjects collection while it is in a loop
		protected var newBaseObjects:ArrayCollection = new ArrayCollection();
		// a collection where removed BaseObjects are placed, to avoid removing items
		// to baseObjects while in the baseObjects collection while it is in a loop
		protected var removedBaseObjects:ArrayCollection = new ArrayCollection();
		// the last frame time
		protected var lastFrame:Date;
		// this flag is set to true to indicate that the ApplicationManager has been initialised
		protected var applicationManagerStarted:Boolean = false;


		public function EngineManager()
		{
			super();
			addEventListener(Event.ADDED_TO_STAGE, init);
		}

		public function init(e:Event): void
		{
			stage.scaleMode = StageScaleMode.NO_SCALE;
			stage.align = StageAlign.TOP_LEFT;

			// Creating scene
			scene = new Scene3D();
			scene.root = new Object3D();
			scene.splitAnalysis = false;

			// Adding camera and view
			camera = new Camera3D();
			camera.x = 0;
			camera.y = 0;
			camera.z = 100;
			scene.root.addChild(camera);

			view = new View();
			addChild(view);
			view.camera = camera;
			view.interactive = true;

			// Connecting camera controller
			cameraController = new CameraController(stage);
			cameraController.camera = camera;
			cameraController.setDefaultBindings();
			cameraController.checkCollisions = true;
			cameraController.collisionRadius = 20;
			cameraController.controlsEnabled = false;
			cameraController.lookAt(new Point3D());

			// FPS display launch
			FPS.init(stage);

			stage.addEventListener(Event.RESIZE, onResize);
			stage.addEventListener(Event.ENTER_FRAME, onEnterFrame);
			onResize(null);

			// set the initial frame time
			lastFrame = new Date();

			// load the resources
			ResourceManager.loadResources();
		}

		private function onResize(e:Event):void
		{
			view.width = stage.stageWidth;
			view.height = stage.stageHeight;
			Application.application.width = stage.stageWidth;
			Application.application.height = stage.stageHeight;
		}

		/**
		 * 	This function is called once per frame. It notfies the BaseObjects
		 *  that a frame is about to be rendered and then calculates the scene
		 */
		protected function onEnterFrame(event:Event):void
		{
			// first check to make sure that all the resources have been loaded
			if (ResourceManager.allResourcesLoaded)
			{
				// once the resources have been loaded we can initiliase the ApplicationManager
				if (!applicationManagerStarted)
				{
					// set the flag to prevent the ApplicationManager being initialised twice
					applicationManagerStarted = true;
					// start the application
					ApplicationManager.Instance.startupApplicationManager();
				}

				// Calculate the time since the last frame
				var thisFrame:Date = new Date();
				var seconds:Number = (thisFrame.getTime() - lastFrame.getTime())/1000.0;
		    	lastFrame = thisFrame;

		    	// sync the baseObjects collection with any BaseObjects created or removed during the
		    	// render loop
		    	removeDeletedBaseObjects();
		    	insertNewBaseObjects();

		    	// allow each BaseObject to update itself
		    	for each (var baseObject:BaseObject in baseObjects)
		    		baseObject.enterFrame(seconds);

				// Scene calculating
				scene.calculate();
			}
		}

		/**
		 * 	Adds a BaseObject to the temporary collection newBaseObjects, the
		 *  contents of which will be added to the main baseObjects collection
		 *  in the insertNewBaseObjects function.
		 *
		 *  This function should only be called by the BaseObject class.
		 *
		 *  @param baseObject The BaseObject to be added
		 */
		public function addBaseObject(baseObject:BaseObject):void
		{
			newBaseObjects.addItem(baseObject);
		}

		/**
		 * 	Adds a BaseObject to the temporary collection removedBaseObjects, the
		 *  contents of which will be removed from the main baseObjects collection
		 *  in the removeDeletedBaseObjects function.
		 *
		 *  This function should only be called by the BaseObject class.
		 *
		 * 	@param baseObject The BaseObject to be removed
		 */
		public function removeBaseObject(baseObject:BaseObject):void
		{
			removedBaseObjects.addItem(baseObject);
		}

		/**
		 * 	Shutdown all the BaseObjects
		 */
		protected function shutdownAll():void
		{
			// don't dispose objects twice
			for each (var baseObject:BaseObject in baseObjects)
			{
				var found:Boolean = false;
				for each (var removedObject:BaseObject in removedBaseObjects)
				{
					if (removedObject == baseObject)
					{
						found = true;
						break;
					}
				}

				if (!found)
					baseObject.shutdown();
			}
		}

		/**
		 * 	Shutdown all the BaseObjects
		 */
		public function shutdownAllOfType(type:Class):void
		{
			// don't dispose objects twice
			for each (var baseObject:BaseObject in baseObjects)
			{
				if (baseObject as type != null)
				{
					var found:Boolean = false;
					for each (var removedObject:BaseObject in removedBaseObjects)
					{
						if (removedObject == baseObject)
						{
							found = true;
							break;
						}
					}

					if (!found)
						baseObject.shutdown();
				}
			}
		}

		/**
		 * 	Adds the BaseObjects added to newBaseObjects with the function addBaseObject
		 *  to the main collection
		 */
		protected function insertNewBaseObjects():void
		{
			for each (var baseObject:BaseObject in newBaseObjects)
				baseObjects.addItem(baseObject);

			newBaseObjects.removeAll();
		}

		/**
		 * 	Removes the BaseObjects added to removedBaseObjects with the function removeBaseObject
		 *  from the main collection
		 */
		protected function removeDeletedBaseObjects():void
		{
			for each (var removedObject:BaseObject in removedBaseObjects)
			{
				var i:int = 0;
				for (i = 0; i < baseObjects.length; ++i)
				{
					if (baseObjects.getItemAt(i) == removedObject)
					{
						baseObjects.removeItemAt(i);
						break;
					}
				}

			}

			removedBaseObjects.removeAll();
		}
	}
}