﻿package game {
	
	// Flash items
	import flash.display.Sprite;
	import flash.events.KeyboardEvent;
	import flash.events.TimerEvent;
	import flash.ui.Keyboard;
	import flash.utils.Timer;
	import flash.events.Event;
	import flash.events.MouseEvent;
	import jiglib.math.JMatrix3D;
	
	// JiglibFlash items
	import jiglib.geometry.JBox;
	import jiglib.geometry.JCapsule;
	import jiglib.geometry.JSphere;
	import jiglib.math.JNumber3D;
	import jiglib.physics.PhysicsSystem;
	import jiglib.plugin.papervision3d.Papervision3DPhysics;
	import jiglib.plugin.papervision3d.Pv3dMesh;
	
	// Papervision3D items
	import org.papervision3d.cameras.Camera3D;
	import org.papervision3d.lights.PointLight3D;
	import org.papervision3d.materials.BitmapMaterial;
	import org.papervision3d.materials.shadematerials.GouraudMaterial;
	import org.papervision3d.objects.DisplayObject3D;
	import org.papervision3d.objects.primitives.Cylinder;
	import org.papervision3d.objects.primitives.Plane;
	import org.papervision3d.objects.primitives.Sphere;
	import org.papervision3d.render.BasicRenderEngine;
	import org.papervision3d.scenes.Scene3D;
	import org.papervision3d.view.Viewport3D;
	
	/**
	 * 3D Bowling game with physics engine
	 * @author Devon O Wolfgang
	 */
	public class BowlingGame extends Sprite {
		
		[Embed(source = "../../assets/lane_material.jpg")]
		private var LaneImage:Class;
		
		[Embed(source = "../../assets/ball_material.jpg")]
		private var BallImage:Class;
		
		private const NUM_PINS:int = 10;
		private const ROLL_SECONDS:int = 7;
		
		private var _view:Viewport3D;
		private var _scene:Scene3D;
		private var _camera:Camera3D;
		private var _engine:BasicRenderEngine;
		private var _light:PointLight3D;
		private var _physics:Papervision3DPhysics;
		
		private var _visualPins:Vector.<Cylinder> = new Vector.<Cylinder>();
		private var _physicsPins:Vector.<BowlingPin> = new Vector.<BowlingPin>();
		private var _pinPositions:Vector.<JNumber3D> = new Vector.<JNumber3D>();
		
		private var _physicsBall:BowlingBall;
		private var _ballIsSet:Boolean = false;
		private var _ballIsRolling:Boolean = false;
		
		private var _rollTimer:Timer;
		
		private var _currentRoll:int = 0;
		
		private var _moveCamRight:Boolean = false;
		private var _moveCamLeft:Boolean = false;
		private var _moveCamUp:Boolean = false;
		private var _moveCamDown:Boolean = false;
		private var _moveAmount:int = 10;
		
		public function BowlingGame() {
			addEventListener(Event.ADDED_TO_STAGE, init);
		}
		
		// initialize a few properties and call the methods that initialize everything else
		private function init(event:Event):void {
			removeEventListener(Event.ADDED_TO_STAGE, init);
			
			_rollTimer = new Timer(ROLL_SECONDS * 1000, 1);
			_rollTimer.addEventListener(TimerEvent.TIMER_COMPLETE, endRoll);
			
			// these positions were determined by trial and error - no magic here
			_pinPositions.push(new JNumber3D(0, 41, 350), new JNumber3D( -35, 41, 390), new JNumber3D(35, 41, 390), new JNumber3D(-70, 41, 430), new JNumber3D(0, 41, 430), new JNumber3D(70, 41, 430), new JNumber3D( -105, 41, 470), new JNumber3D( -35, 41, 470), new JNumber3D(35, 41, 470), new JNumber3D(105, 41, 470));
			
			init3d();
			initLane();	
			initPins();
			initBall();
			initRollerUI();
			
			stage.addEventListener(KeyboardEvent.KEY_DOWN,keyDownHandler);
			stage.addEventListener(KeyboardEvent.KEY_UP, keyUpHandler);
			stage.addEventListener(MouseEvent.MOUSE_WHEEL, mouseWheelHandler);
			
			startRender();
		}
		
		// to move the camera around
		private function keyDownHandler(event:KeyboardEvent):void {
			if (event.keyCode == Keyboard.UP) _moveCamUp = true;
			if (event.keyCode == Keyboard.DOWN) _moveCamDown = true;
			if (event.keyCode == Keyboard.LEFT) _moveCamLeft = true;
			if (event.keyCode == Keyboard.RIGHT) _moveCamRight = true;
		}
		
		// to stop moving the camera around
		private function keyUpHandler(event:KeyboardEvent):void {
			if (event.keyCode == Keyboard.UP) _moveCamUp = false;
			if (event.keyCode == Keyboard.DOWN) _moveCamDown = false;
			if (event.keyCode == Keyboard.LEFT) _moveCamLeft = false;
			if (event.keyCode == Keyboard.RIGHT) _moveCamRight = false;
		}
		
		// to move the camera forwards and backwards
		private function mouseWheelHandler(event:MouseEvent):void {
			_camera.z += _moveAmount * event.delta;
			if (_camera.z < -1500) _camera.z = -1500;
			if (_camera.z > -200) _camera.z = -200;
		}
		
		// initialize the papervision3d elements and the jiglibflash physics engine
		private function init3d():void {
			_scene = new Scene3D();
			_camera = new Camera3D();
			_camera.zoom = 100;
			_camera.y = 200;
			_engine = new BasicRenderEngine();
			_light = new PointLight3D();
			_light.y = 400;
			_light.z = -500;
			_view = new Viewport3D(stage.stageWidth, stage.stageHeight);
			addChild(_view);
			
			_physics = new Papervision3DPhysics(_scene, 3);
		}
		
		// initialize the visual and physics elements of the lane
		private function initLane():void {
			var laneMat:BitmapMaterial = new BitmapMaterial(new LaneImage().bitmapData);
			var visualLane:Plane = new Plane(laneMat, 250, 1000, 8, 32);
			visualLane.rotationX = 90;
			_camera.target = visualLane;
			_scene.addChild(visualLane);
			
			var physicsLane:JBox = new JBox(new Pv3dMesh(visualLane), 250, 1, 1000);
			_physics.addBody(physicsLane);
			physicsLane.movable = false;
			physicsLane.rotationX = 90;
		}
		
		// initialize the visual and physics elements of the pins
		private function initPins():void {
			var mat:GouraudMaterial = new GouraudMaterial(_light);
			
			for (var i:int = 0; i < NUM_PINS; i++) {	
				var visualPin:Cylinder = new Cylinder(mat, 10, 75);
				visualPin.name = String("pin_" + i);
				_visualPins.push(visualPin);
				_scene.addChild(visualPin);
				
				var physicsPin:BowlingPin = new BowlingPin(new Pv3dMesh(visualPin), 10, 55);
				physicsPin.resetPosition = _pinPositions[i];
				physicsPin.reset();
				physicsPin.movable = true;
				physicsPin.mass = 3;
				_physics.addBody(physicsPin);
				_physicsPins.push(physicsPin);
			}
		}
		
		// initialize the visual and physics elements of the ball
		private function initBall():void {
			var ballMat:BitmapMaterial = new BitmapMaterial(new BallImage().bitmapData);
			var visualBall:Sphere = new Sphere(ballMat, 20);
			_scene.addChild(visualBall);
			
			_physicsBall = new BowlingBall(new Pv3dMesh(visualBall), 20);
			_physicsBall.resetPosition = new JNumber3D(0, 50, -450);
			_physics.addBody(_physicsBall);
			resetBall();
		}
		
		// initialize the gui elements that are used to position and roll the ball
		private function initRollerUI():void {
			var ballSetter:Sprite = new Sprite();
			ballSetter.graphics .beginFill(0x006600);
			ballSetter.graphics.drawRect(0, 0, 100, 10);
			ballSetter.graphics.endFill();
			ballSetter.addEventListener(MouseEvent.CLICK, setBall);
			ballSetter.x = 4;
			ballSetter.y = 4;
			addChild(ballSetter);
			
			var ballRoller:Sprite = new Sprite();
			ballRoller.graphics.beginFill(0x660000);
			ballRoller.graphics.drawRect(0, 0, 100, 100);
			ballRoller.graphics.endFill();
			ballRoller.addEventListener(MouseEvent.CLICK, rollBall);
			ballRoller.x = 4;
			ballRoller.y = ballSetter.y + ballSetter.height + 4;
			addChild(ballRoller);
		}
		
		// position the ball on the x axis
		private function setBall(event:MouseEvent):void {
			if (!_ballIsSet && !_ballIsRolling) {
				_ballIsSet = true;
				
				var xpos:Number = ((event.currentTarget.mouseX / event.currentTarget.width) - .5) * 2;
				xpos *= 100;
				
				_physicsBall.moveTo(new JNumber3D(xpos, 20, -450));
			}
		}
		
		// roll the ball
		private function rollBall(event:MouseEvent):void {
			if (_ballIsSet && !_ballIsRolling) {
				_ballIsRolling = true;
				
				var xforce:Number = ((event.currentTarget.mouseX / event.currentTarget.width) - .5) * 2;
				xforce *= 500;
				
				var zforce:Number = (event.currentTarget.mouseY / event.currentTarget.height) - 1;
				zforce *= -850;
				
				_physicsBall.addWorldForce(new JNumber3D(xforce, 0, zforce), _physicsBall.currentState.position);
				
				_rollTimer.start();
			}
		}
		
		// called 7 seconds after the ball is rolled by default
		private function endRoll(event:TimerEvent):void {
			_rollTimer.reset();
			_currentRoll++;
			resetBall();
			clearPins();
			_ballIsRolling = false;
			_ballIsSet = false;
		}
		
		// move the ball back to its reset position
		private function resetBall():void {
			_physicsBall.reset();
		}
		
		// remove the pins that have been knocked over
		private function clearPins():void {
			var numPinsDown:int = 0;
			for (var i:int = 0; i < NUM_PINS; i++) {
				if (_physicsPins[i].isDown) {
					_physics.removeBody(_physicsPins[i]);
					_scene.removeChild(_visualPins[i]);
					numPinsDown++;
				}
			}
			
			// if there's a STRIKE, don't roll a second time
			if (numPinsDown == NUM_PINS) _currentRoll = 2;
			
			if (_currentRoll == 2) {
				resetAllPins();
				_currentRoll = 0;
			} else {
				resetPins();
			}
		}
		
		// reset just the pins that are still left standing
		private function resetPins():void {
			for (var i:int = 0; i < NUM_PINS; i++) {
				var p:DisplayObject3D = _scene.getChildByName(String("pin_" + i));
				if (p != null) _physicsPins[i].reset();
			}
		}
		
		// reset all the pins
		private function resetAllPins():void {
			for (var i:int = 0; i < NUM_PINS; i++) {
				var p:DisplayObject3D = _scene.getChildByName(String("pin_" + i));
				if (p == null) {
					p = _visualPins[i] as DisplayObject3D;
					_scene.addChild(p);
					_physics.addBody(_physicsPins[i]);
				}
				_physicsPins[i].reset();
			}
		}
		
		// start the enter frame event that will render the 3D scene
		private function startRender():void {
			addEventListener(Event.ENTER_FRAME, render);
		}
		
		// move the camera, update the physics engine, then render the 3D scene
		private function render(event:Event):void {
			
			if (_moveCamRight) {
				if (_camera.x < 1000) _camera.x += _moveAmount;
			}
			
			if (_moveCamLeft) {
				if (_camera.x > -1000) _camera.x -= _moveAmount;
			}
			
			if (_moveCamUp) {
				if (_camera.y < 500) _camera.y += _moveAmount;
			}
			
			if (_moveCamDown) {
				if (_camera.y > 50) _camera.y -= _moveAmount;
			}
			
			PhysicsSystem.getInstance().integrate(.2);
			// Could use
			//_physics.step();
			// instead
			
			_engine.renderScene(_scene, _camera, _view);
		}
	}
}